/**
 * Sistema de Validação e API para Frontend
 * Versão melhorada com validações robustas e tratamento de erros
 */

class ComissoesAPI {
    constructor() {
        this.baseURL = '';
        this.token = localStorage.getItem('access_token');
        this.user = JSON.parse(localStorage.getItem('user') || '{}');
    }

    // Método para fazer requisições HTTP com tratamento de erros
    async request(endpoint, options = {}) {
        const url = `${this.baseURL}${endpoint}`;
        const token = localStorage.getItem('access_token'); // Sempre pegar token fresco
        const config = {
            headers: {
                'Content-Type': 'application/json',
                ...(token && { 'Authorization': `Bearer ${token}` })
            },
            ...options
        };

        try {
            console.log('🔗 Fazendo requisição para:', url, 'com config:', config);
            const response = await fetch(url, config);
            console.log('📊 Status da resposta:', response.status, response.statusText);
            const data = await response.json();
            console.log('📋 Dados da resposta:', data);

            if (!response.ok) {
                throw new Error(data.msg || `Erro HTTP: ${response.status}`);
            }

            return { success: true, data };
        } catch (error) {
            console.error('❌ Erro na requisição:', error);
            console.error('❌ URL:', url, 'Config:', config);
            return { 
                success: false, 
                error: error.message || 'Erro de conexão'
            };
        }
    }

    // Métodos específicos da API
    async login(email, senha) {
        return await this.request('/auth/login', {
            method: 'POST',
            body: JSON.stringify({ email, senha })
        });
    }

    async getColaboradores(departamento = null) {
        const params = departamento ? `?departamento=${encodeURIComponent(departamento)}` : '';
        const result = await this.request(`/admin/colaboradores${params}`);
        return result.success ? result.data : [];
    }

    async addComissao(dadosComissao) {
        return await this.request('/comissoes/add', {
            method: 'POST',
            body: JSON.stringify(dadosComissao)
        });
    }

    async getHistoricoComissoes(mes = null) {
        const params = mes ? `?mes=${mes}` : '';
        return await this.request(`/comissoes/historico${params}`);
    }

    async updateComissao(id, dados) {
        return await this.request(`/comissoes/${id}`, {
            method: 'PUT',
            body: JSON.stringify(dados)
        });
    }
}

class Validators {
    static email(email) {
        const pattern = /^[^\s@]+@autonunes\.com\.br$/i;
        if (!email) return { valid: false, message: 'Email é obrigatório' };
        if (!pattern.test(email)) return { valid: false, message: 'Email deve ser @autonunes.com.br' };
        return { valid: true };
    }

    static senha(senha) {
        if (!senha) return { valid: false, message: 'Senha é obrigatória' };
        if (senha.length < 6) return { valid: false, message: 'Senha deve ter pelo menos 6 caracteres' };
        return { valid: true };
    }

    static valorMonetario(valor, campo) {
        if (valor === '' || valor === null || valor === undefined) return { valid: true }; // Opcional
        
        const num = parseFloat(valor);
        if (isNaN(num)) return { valid: false, message: `${campo} deve ser um número válido` };
        if (num < 0) return { valid: false, message: `${campo} não pode ser negativo` };
        if (num > 999999.99) return { valid: false, message: `${campo} muito alto (máx: 999.999,99)` };
        
        return { valid: true };
    }

    static mesReferencia(mes) {
        if (!mes) return { valid: false, message: 'Mês de referência é obrigatório' };
        
        const pattern = /^\d{4}-\d{2}-\d{2}$/;
        if (!pattern.test(mes)) return { valid: false, message: 'Formato de data inválido' };
        
        const data = new Date(mes);
        const hoje = new Date();
        const umAnoAtras = new Date();
        umAnoAtras.setFullYear(hoje.getFullYear() - 1);
        
        if (data > hoje) return { valid: false, message: 'Data não pode ser futura' };
        if (data < umAnoAtras) return { valid: false, message: 'Data muito antiga (máx: 1 ano)' };
        
        return { valid: true };
    }

    static dataReferencia(data) {
        if (!data) return { valid: false, message: 'Data de referência é obrigatória' };
        
        const pattern = /^\d{4}-\d{2}-\d{2}$/;
        if (!pattern.test(data)) return { valid: false, message: 'Formato de data inválido (AAAA-MM-DD)' };
        
        const dataObj = new Date(data + 'T00:00:00'); // Evitar problemas de timezone
        const hoje = new Date();
        const umAnoAtras = new Date();
        umAnoAtras.setFullYear(hoje.getFullYear() - 1);
        
        if (isNaN(dataObj.getTime())) return { valid: false, message: 'Data inválida' };
        if (dataObj > hoje) return { valid: false, message: 'Data não pode ser futura' };
        if (dataObj < umAnoAtras) return { valid: false, message: 'Data muito antiga (máx: 1 ano)' };
        
        return { valid: true };
    }

    static vendedorId(id) {
        if (!id) return { valid: false, message: 'Colaborador é obrigatório' };
        
        const num = parseInt(id);
        if (isNaN(num) || num <= 0) return { valid: false, message: 'Colaborador inválido' };
        
        return { valid: true };
    }
}

class UIHelpers {
    static showMessage(elementId, message, type = 'info') {
        const element = document.getElementById(elementId);
        if (!element) return;

        const alertClass = {
            'success': 'alert-success',
            'error': 'alert-danger',
            'warning': 'alert-warning',
            'info': 'alert-info'
        }[type] || 'alert-info';

        element.innerHTML = `
            <div class="alert ${alertClass} alert-dismissible fade show" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `;

        // Auto-remove após 5 segundos
        setTimeout(() => {
            element.innerHTML = '';
        }, 5000);
    }

    static setLoading(buttonId, loading = true) {
        const button = document.getElementById(buttonId);
        if (!button) return;

        if (loading) {
            button.disabled = true;
            button.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Carregando...';
        } else {
            button.disabled = false;
            button.innerHTML = button.getAttribute('data-original-text') || 'Salvar';
        }
    }

    static formatCurrency(value) {
        if (!value) return '-';
        return new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(value);
    }

    static formatDate(dateString) {
        if (!dateString) return '-';
        
        // Criar uma nova data e ajustar para o timezone brasileiro (UTC-3)
        const date = new Date(dateString);
        
        // Usar toLocaleString com timezone brasileiro
        return date.toLocaleString('pt-BR', {
            timeZone: 'America/Sao_Paulo',
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit',
            second: '2-digit'
        });
    }

    static addFieldValidation(fieldId, validator) {
        const field = document.getElementById(fieldId);
        if (!field) return;

        const showValidation = (result) => {
            field.classList.remove('is-valid', 'is-invalid');
            
            let feedbackElement = field.parentNode.querySelector('.invalid-feedback');
            if (!feedbackElement) {
                feedbackElement = document.createElement('div');
                feedbackElement.className = 'invalid-feedback';
                field.parentNode.appendChild(feedbackElement);
            }

            if (result.valid) {
                field.classList.add('is-valid');
                feedbackElement.style.display = 'none';
            } else {
                field.classList.add('is-invalid');
                feedbackElement.textContent = result.message;
                feedbackElement.style.display = 'block';
            }
        };

        field.addEventListener('blur', () => {
            const result = validator(field.value);
            showValidation(result);
        });

        field.addEventListener('input', () => {
            if (field.classList.contains('is-invalid')) {
                const result = validator(field.value);
                if (result.valid) {
                    showValidation(result);
                }
            }
        });
    }
}

// Instância global da API
window.comissoesAPI = new ComissoesAPI();
window.Validators = Validators;
window.UIHelpers = UIHelpers;