// Classe para gerenciar o histórico de comissões
class HistoricoComissoes {
    constructor(api) {
        this.api = api;
        
        // Propriedades de paginação
        this.currentPage = 1;
        this.itemsPerPage = 2;
        this.totalItems = 0;
        this.totalPages = 0;
        this.allComissoes = [];
        this.dadosOriginais = [];
    }

    async carregarColaboradores() {
        try {
            console.log('Carregando colaboradores...');
            let colaboradores;
            
            if (this.api) {
                // Usar a instância da API se disponível
                const result = await this.api.getColaboradores();
                colaboradores = result.success ? result.data : result;
            } else {
                // Fallback para fetch direto
                const token = localStorage.getItem('access_token');
                if (!token) {
                    window.location.href = '/frontend/login.html';
                    return;
                }

                const response = await fetch('/admin/colaboradores', {
                    method: 'GET',
                    headers: {
                        'Authorization': `Bearer ${token}`,
                        'Content-Type': 'application/json'
                    }
                });

                if (!response.ok) {
                    throw new Error('Erro ao carregar colaboradores');
                }

                colaboradores = await response.json();
            }
            
            console.log('Resultado colaboradores:', colaboradores);
            this.preencherSelectVendedores(colaboradores);
            
        } catch (error) {
            console.error('Erro ao carregar colaboradores:', error);
        }
    }

    preencherSelectVendedores(colaboradores) {
        const select = document.getElementById('filtroVendedor');
        if (!select) {
            console.error('❌ Elemento select filtroVendedor não encontrado');
            return;
        }

        // Limpar opções existentes (exceto a primeira)
        select.innerHTML = '<option value="">Todos os vendedores</option>';

        // Verificar se colaboradores é um array válido
        if (!Array.isArray(colaboradores)) {
            console.error('❌ Colaboradores não é um array válido:', colaboradores);
            return;
        }

        // Adicionar colaboradores
        colaboradores.forEach(colaborador => {
            const option = document.createElement('option');
            option.value = colaborador.id;
            option.textContent = `${colaborador.nome} - ${colaborador.departamento}`;
            select.appendChild(option);
        });

        console.log(`✅ ${colaboradores.length} colaboradores carregados no select`);
    }

    async carregarHistorico(mes = null, resetPage = true) {
        try {
            const lista = document.getElementById('lista-comissoes');
            if (!lista) {
                console.error('Elemento lista-comissoes não encontrado');
                return;
            }

            lista.innerHTML = '<div class="text-center py-8"><div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div><p class="mt-2 text-gray-600">Carregando...</p></div>';

            console.log('Chamando API para histórico com mês:', mes);
            const result = await this.api.getHistoricoComissoes(mes);
            console.log('Resultado da API:', result);

            if (result.success) {
                console.log('Dados recebidos:', result.data);
                
                // Armazenar todas as comissões
                this.allComissoes = result.data || [];
                // Armazenar dados originais para filtros se ainda não temos
                if (this.dadosOriginais.length === 0) {
                    this.dadosOriginais = [...this.allComissoes];
                }
                this.totalItems = this.allComissoes.length;
                this.totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
                
                // Resetar para primeira página se necessário
                if (resetPage) {
                    this.currentPage = 1;
                }
                
                // Renderizar página atual
                this.renderizarPaginaAtual();
                
            } else {
                console.log('Erro na API:', result.error);
                lista.innerHTML = `<div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
                    <strong class="font-bold">Erro!</strong>
                    <span class="block sm:inline">${result.error}</span>
                </div>`;
            }

        } catch (error) {
            console.error('Erro ao carregar histórico:', error);
            const lista = document.getElementById('lista-comissoes');
            if (lista) {
                lista.innerHTML = '<div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">Erro ao carregar histórico</div>';
            }
        }
    }

    renderizarPaginaAtual() {
        const container = document.getElementById('lista-comissoes');
        if (!container) return;

        // Calcular índices da página atual
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const comissoesPagina = this.allComissoes.slice(startIndex, endIndex);

        console.log(`Renderizando página ${this.currentPage}: ${startIndex}-${endIndex}`, comissoesPagina);

        if (comissoesPagina.length === 0) {
            container.innerHTML = '<div class="bg-blue-100 border border-blue-400 text-blue-700 px-4 py-3 rounded relative text-center" role="alert">Nenhuma comissão encontrada</div>';
            this.renderizarControlesPaginacao();
            return;
        }

        this.renderizarHistorico(comissoesPagina, container);
        this.renderizarControlesPaginacao();
    }

    renderizarHistorico(comissoes, container) {
        console.log('renderizarHistorico chamado com:', comissoes);
        console.log('Container:', container);
        
        if (!comissoes || comissoes.length === 0) {
            console.log('Nenhuma comissão encontrada');
            container.innerHTML = '<div class="bg-blue-100 border border-blue-400 text-blue-700 px-4 py-3 rounded relative text-center" role="alert">Nenhuma comissão encontrada</div>';
            return;
        }

        console.log(`Renderizando ${comissoes.length} comissões`);
        container.innerHTML = '';

        comissoes.forEach((comissao, index) => {
            console.log(`Renderizando comissão ${index}:`, comissao);
            const div = document.createElement('div');
            div.className = 'bg-white border border-gray-200 rounded-lg shadow-sm mb-4 p-4';
            
            const dataRegistro = new Date(comissao.data_criacao);
            const agora = new Date();
            const diffHoras = (agora - dataRegistro) / (1000 * 60 * 60);
            const podeEditar = diffHoras <= 1; // 1 hora para edição

            div.innerHTML = `
                <div class="flex flex-col md:flex-row md:justify-between">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-800 mb-2">${comissao.colaborador_nome || 'Colaborador'}</h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                            <div>
                                <span class="font-medium text-gray-600">Mês:</span>
                                <span class="text-gray-800">${comissao.mes_referencia}</span>
                            </div>
                            <div>
                                <span class="font-medium text-gray-600">PREMIO ISC:</span>
                                <span class="text-green-600 font-semibold">${UIHelpers.formatCurrency(comissao.PREMIO_ISC)}</span>
                            </div>
                            <div>
                                <span class="font-medium text-gray-600">PREMIAÇÃO META:</span>
                                <span class="text-green-600 font-semibold">${UIHelpers.formatCurrency(comissao.PREMIACAO_META)}</span>
                            </div>
                            <div>
                                <span class="font-medium text-gray-600">PREMIO:</span>
                                <span class="text-green-600 font-semibold">${UIHelpers.formatCurrency(comissao.PREMIO)}</span>
                            </div>
                            <div>
                                <span class="font-medium text-gray-600">GRATIFICAÇÃO:</span>
                                <span class="text-green-600 font-semibold">${UIHelpers.formatCurrency(comissao.GRATIFICACAO)}</span>
                            </div>
                        </div>
                        <div class="mt-3 text-xs text-gray-500">
                            <i class="fas fa-clock mr-1"></i>
                            Registrado em: ${UIHelpers.formatDate(comissao.data_criacao)}
                        </div>
                    </div>
                    <div class="mt-4 md:mt-0 md:ml-4 flex-shrink-0">
                        ${podeEditar ? this.renderBotoesEdicao(comissao) : this.renderAvisoExpirado()}
                    </div>
                </div>
            `;

            container.appendChild(div);
        });
    }

    renderBotoesEdicao(comissao) {
        return `
            <button class="bg-blue-500 hover:bg-blue-600 text-white px-3 py-1 rounded text-sm transition duration-200" onclick="editarComissao(${comissao.id})">
                <i class="fas fa-edit mr-1"></i> Editar
            </button>
            <div id="edit-area-${comissao.id}" style="display:none;" class="mt-3 p-3 bg-gray-50 rounded">
                <div class="space-y-2">
                    <div>
                        <label class="block text-xs font-medium text-gray-700">PREMIO ISC:</label>
                        <input type="number" step="0.01" class="w-full px-2 py-1 text-sm border border-gray-300 rounded focus:outline-none focus:ring-1 focus:ring-primary" 
                               value="${comissao.PREMIO_ISC || ''}" id="edit-isc-${comissao.id}">
                    </div>
                    <div>
                        <label class="block text-xs font-medium text-gray-700">PREMIAÇÃO META:</label>
                        <input type="number" step="0.01" class="w-full px-2 py-1 text-sm border border-gray-300 rounded focus:outline-none focus:ring-1 focus:ring-primary" 
                               value="${comissao.PREMIACAO_META || ''}" id="edit-meta-${comissao.id}">
                    </div>
                    <div>
                        <label class="block text-xs font-medium text-gray-700">PREMIO:</label>
                        <input type="number" step="0.01" class="w-full px-2 py-1 text-sm border border-gray-300 rounded focus:outline-none focus:ring-1 focus:ring-primary" 
                               value="${comissao.PREMIO || ''}" id="edit-premio-${comissao.id}">
                    </div>
                    <div>
                        <label class="block text-xs font-medium text-gray-700">GRATIFICAÇÃO:</label>
                        <input type="number" step="0.01" class="w-full px-2 py-1 text-sm border border-gray-300 rounded focus:outline-none focus:ring-1 focus:ring-primary" 
                               value="${comissao.GRATIFICACAO || ''}" id="edit-grat-${comissao.id}">
                    </div>
                    <div class="flex space-x-2 pt-2">
                        <button class="bg-green-500 hover:bg-green-600 text-white px-3 py-1 rounded text-sm" onclick="salvarEdicao(${comissao.id})">Salvar</button>
                        <button class="bg-gray-500 hover:bg-gray-600 text-white px-3 py-1 rounded text-sm" onclick="cancelarEdicao(${comissao.id})">Cancelar</button>
                    </div>
                </div>
                <div id="msg-edit-${comissao.id}" class="mt-2"></div>
            </div>
        `;
    }

    renderAvisoExpirado() {
        return '<div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-3 py-1 rounded text-sm">Prazo de edição expirado (1h)</div>';
    }

    renderizarControlesPaginacao() {
        const container = document.getElementById('controles-paginacao');
        if (!container) return;

        if (this.totalPages <= 1) {
            container.innerHTML = '';
            return;
        }

        let html = '<nav aria-label="Navegação de páginas" class="mt-6">';
        html += '<div class="flex justify-center items-center space-x-1">';

        // Botão anterior
        html += `<button class="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-l-md hover:bg-gray-50 ${this.currentPage === 1 ? 'cursor-not-allowed opacity-50' : 'hover:text-gray-700'}" 
                        ${this.currentPage === 1 ? 'disabled' : ''} 
                        onclick="historico.irParaPagina(${this.currentPage - 1})">
                    <i class="fas fa-chevron-left"></i> Anterior
                 </button>`;

        // Números das páginas
        for (let i = 1; i <= this.totalPages; i++) {
            const isActive = i === this.currentPage;
            html += `<button class="px-3 py-2 text-sm font-medium ${isActive ? 'text-white bg-primary border-primary' : 'text-gray-500 bg-white border-gray-300 hover:bg-gray-50 hover:text-gray-700'} border" 
                            onclick="historico.irParaPagina(${i})">
                        ${i}
                     </button>`;
        }

        // Botão próximo
        html += `<button class="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-r-md hover:bg-gray-50 ${this.currentPage === this.totalPages ? 'cursor-not-allowed opacity-50' : 'hover:text-gray-700'}" 
                        ${this.currentPage === this.totalPages ? 'disabled' : ''} 
                        onclick="historico.irParaPagina(${this.currentPage + 1})">
                    Próximo <i class="fas fa-chevron-right"></i>
                 </button>`;

        html += '</div>';
        html += `<div class="text-center text-gray-500 text-sm mt-3">
                    Página ${this.currentPage} de ${this.totalPages} 
                    (${this.totalItems} comissões no total)
                 </div>`;
        html += '</nav>';

        container.innerHTML = html;
    }

    irParaPagina(pagina) {
        if (pagina < 1 || pagina > this.totalPages) return;
        
        this.currentPage = pagina;
        this.renderizarPaginaAtual();
    }

    async filtrarHistorico() {
        const mes = document.getElementById('filtroMes').value;
        const vendedorId = document.getElementById('filtroVendedor').value;
        
        console.log('🔍 Filtrando por:', { mes, vendedorId });

        try {
            // Se ainda não carregamos os dados originais, carregar
            if (this.dadosOriginais.length === 0) {
                console.log('Carregando dados originais...');
                const result = await this.api.getHistoricoComissoes();
                if (result.success) {
                    this.dadosOriginais = result.data || [];
                }
            }

            // Começar com todos os dados
            let dadosFiltrados = [...this.dadosOriginais];

            // Aplicar filtro por mês se selecionado
            if (mes) {
                dadosFiltrados = dadosFiltrados.filter(comissao => {
                    const dataComissao = new Date(comissao.mes_referencia);
                    const mesAno = `${dataComissao.getFullYear()}-${String(dataComissao.getMonth() + 1).padStart(2, '0')}`;
                    return mesAno === mes;
                });
                console.log(`Após filtro mês: ${dadosFiltrados.length} registros`);
            }

            // Aplicar filtro por vendedor se selecionado  
            if (vendedorId) {
                dadosFiltrados = dadosFiltrados.filter(comissao => {
                    return comissao.vendedor_id.toString() === vendedorId;
                });
                console.log(`Após filtro vendedor: ${dadosFiltrados.length} registros`);
            }

            console.log(`✅ Total após filtros: ${dadosFiltrados.length} registros`);

            // Atualizar dados filtrados
            this.allComissoes = dadosFiltrados;
            this.totalItems = this.allComissoes.length;
            this.totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
            this.currentPage = 1;

            // Renderizar resultados
            this.renderizarPaginaAtual();
            
        } catch (error) {
            console.error('💥 Erro ao filtrar:', error);
            const lista = document.getElementById('lista-comissoes');
            if (lista) {
                lista.innerHTML = '<div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">Erro ao aplicar filtros</div>';
            }
        }
    }

    limparFiltro() {
        console.log('🧹 Limpando filtros...');
        
        // Limpar campos de filtro
        document.getElementById('filtroMes').value = '';
        document.getElementById('filtroVendedor').value = '';
        
        // Restaurar dados originais
        this.allComissoes = [...this.dadosOriginais];
        this.totalItems = this.allComissoes.length;
        this.totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
        this.currentPage = 1;
        
        // Renderizar todos os dados
        this.renderizarPaginaAtual();
        
        console.log('✅ Filtros limpos, mostrando todos os dados');
    }

    limparFiltro() {
        document.getElementById('filtroMes').value = '';
        document.getElementById('filtroVendedor').value = '';
        this.carregarHistorico(null, true); // Reset page quando limpar filtro
    }

    async salvarEdicao(comissaoId) {
        try {
            const dados = {
                PREMIO_ISC: this.parseFloat(document.getElementById(`edit-isc-${comissaoId}`).value),
                PREMIACAO_META: this.parseFloat(document.getElementById(`edit-meta-${comissaoId}`).value),
                PREMIO: this.parseFloat(document.getElementById(`edit-premio-${comissaoId}`).value),
                GRATIFICACAO: this.parseFloat(document.getElementById(`edit-grat-${comissaoId}`).value)
            };

            const result = await this.api.updateComissao(comissaoId, dados);

            if (result.success) {
                UIHelpers.showMessage(`msg-edit-${comissaoId}`, 'Comissão atualizada com sucesso!', 'success');
                setTimeout(() => this.carregarHistorico(null, false), 1000); // Manter página atual
            } else {
                UIHelpers.showMessage(`msg-edit-${comissaoId}`, result.error, 'error');
            }

        } catch (error) {
            console.error('Erro ao salvar edição:', error);
            UIHelpers.showMessage(`msg-edit-${comissaoId}`, 'Erro interno. Tente novamente.', 'error');
        }
    }

    parseFloat(value) {
        if (!value || value.trim() === '') return null;
        const num = parseFloat(value);
        return isNaN(num) ? null : num;
    }
}

// Funções globais para edição
function editarComissao(comissaoId) {
    const editArea = document.getElementById(`edit-area-${comissaoId}`);
    const editBtn = editArea.previousElementSibling;
    
    if (editArea.style.display === 'none') {
        editArea.style.display = 'block';
        editBtn.style.display = 'none';
    }
}

function cancelarEdicao(comissaoId) {
    const editArea = document.getElementById(`edit-area-${comissaoId}`);
    const editBtn = editArea.previousElementSibling;
    
    editArea.style.display = 'none';
    editBtn.style.display = 'inline-block';
}

function salvarEdicao(comissaoId) {
    if (window.historico) {
        window.historico.salvarEdicao(comissaoId);
    }
}