/**
 * Funções específicas para Registro de Comissão
 * Versão melhorada com validações e tratamento de erros
 */

class RegistroComissao {
    constructor(api = null) {
        this.api = api;
        this.form = document.getElementById('form-comissao');
        this.setupValidations();
        this.setupEventListeners();
    }

    // Função removida - agora usamos busca inteligente no HTML

    setupValidations() {
        // Adicionar validações aos campos
        UIHelpers.addFieldValidation('vendedorId', Validators.vendedorId);
        UIHelpers.addFieldValidation('dataReferencia', Validators.dataReferencia);
        UIHelpers.addFieldValidation('tipo1', (value) => Validators.valorMonetario(value, 'PREMIO ISC'));
        UIHelpers.addFieldValidation('tipo2', (value) => Validators.valorMonetario(value, 'PREMIAÇÃO META'));
        UIHelpers.addFieldValidation('tipo3', (value) => Validators.valorMonetario(value, 'PREMIO'));
        UIHelpers.addFieldValidation('tipo4', (value) => Validators.valorMonetario(value, 'GRATIFICAÇÃO'));
    }

    setupEventListeners() {
        // Configurar botão de salvar
        const btnSalvar = document.getElementById('btn-registrar');
        if (btnSalvar) {
            btnSalvar.setAttribute('data-original-text', 'Salvar Comissão');
            btnSalvar.onclick = () => this.registrarComissao();
        }

        // Configurar input de data para formato correto
        const inputData = document.getElementById('dataReferencia');
        if (inputData) {
            // Já é do tipo date, não precisa alterar
        }
    }

    validateForm() {
        const errors = [];
        
        // Validar vendedor
        const vendedorId = document.getElementById('vendedorId').value;
        const vendedorResult = Validators.vendedorId(vendedorId);
        if (!vendedorResult.valid) errors.push(vendedorResult.message);

        // Validar data
        const dataValue = document.getElementById('dataReferencia').value;
        // Usar a data diretamente sem conversão
        const dataResult = Validators.dataReferencia(dataValue);
        if (!dataResult.valid) errors.push(dataResult.message);

        // Validar valores monetários
        const campos = [
            { id: 'tipo1', name: 'PREMIO ISC' },
            { id: 'tipo2', name: 'PREMIAÇÃO META' },
            { id: 'tipo3', name: 'PREMIO' },
            { id: 'tipo4', name: 'GRATIFICAÇÃO' }
        ];

        campos.forEach(campo => {
            const value = document.getElementById(campo.id).value;
            const result = Validators.valorMonetario(value, campo.name);
            if (!result.valid) errors.push(result.message);
        });

        // Verificar se pelo menos um valor foi preenchido
        const temValor = campos.some(campo => {
            const value = document.getElementById(campo.id).value;
            return value && parseFloat(value) > 0;
        });

        if (!temValor) {
            errors.push('Pelo menos um valor de comissão deve ser preenchido');
        }

        return errors;
    }

    async registrarComissao() {
        try {
            console.log('🚀 Iniciando processo de registro...');
            
            // Validar formulário
            const errors = this.validateForm();
            console.log('🔍 Erros de validação:', errors);
            
            if (errors.length > 0) {
                console.log('❌ Validação falhou, mostrando erros...');
                UIHelpers.showMessage('mensagem-registro', errors.join('<br>'), 'error');
                return;
            }

            console.log('✅ Validação passou, continuando...');

            // Mostrar loading
            UIHelpers.setLoading('btn-registrar', true);

            // Preparar dados
            const dataValue = document.getElementById('dataReferencia').value; // YYYY-MM-DD
            const dados = {
                vendedor_id: document.getElementById('vendedorId').value,
                mes_referencia: dataValue, // Agora é data completa
                PREMIO_ISC: this.parseFloat(document.getElementById('tipo1').value),
                PREMIACAO_META: this.parseFloat(document.getElementById('tipo2').value),
                PREMIO: this.parseFloat(document.getElementById('tipo3').value),
                GRATIFICACAO: this.parseFloat(document.getElementById('tipo4').value)
            };

            // Fazer requisição
            console.log('📤 Enviando dados:', dados);
            const result = await this.api.addComissao(dados);
            console.log('📥 Resposta recebida:', result);

            if (result.success) {
                console.log('✅ Sucesso confirmado, mostrando mensagem...');
                UIHelpers.showMessage('mensagem-registro', 'Comissão registrada com sucesso!', 'success');
                
                // Reset do formulário com verificação de segurança
                if (this.form) {
                    this.form.reset();
                } else {
                    console.warn('⚠️ Formulário não encontrado para reset');
                }
                
                // Recarregar histórico usando instância global
                if (window.historicoComissoes) {
                    console.log('Recarregando histórico após criar comissão...');
                    setTimeout(() => {
                        window.historicoComissoes.carregarHistorico();
                    }, 500);
                }
                
                // Fallback para função global
                if (typeof carregarHistoricoComissoes === 'function') {
                    setTimeout(() => {
                        carregarHistoricoComissoes();
                    }, 500);
                }
            } else {
                console.log('❌ Sucesso = false, mostrando erro...');
                UIHelpers.showMessage('mensagem-registro', result.error, 'error');
            }

        } catch (error) {
            console.error('💥 Erro ao registrar comissão:', error);
            console.error('💥 Tipo do erro:', typeof error);
            console.error('💥 Stack trace:', error.stack);
            UIHelpers.showMessage('mensagem-registro', `Erro interno: ${error.message}`, 'error');
        } finally {
            UIHelpers.setLoading('btn-registrar', false);
        }
    }

    parseFloat(value) {
        if (!value || value.trim() === '') return null;
        const num = parseFloat(value);
        return isNaN(num) ? null : num;
    }
}

class HistoricoComissoes {
    constructor(api = null) {
        this.api = api;
        this.currentPage = 1;
        this.itemsPerPage = 2;
        this.totalItems = 0;
        this.totalPages = 0;
        this.allComissoes = []; // Armazenar todas as comissões
        this.setupEventListeners();
        this.carregarColaboradoresNoFiltro();
    }

    setupEventListeners() {
        const btnFiltrar = document.getElementById('btn-filtrar');
        const btnLimpar = document.getElementById('btn-limpar-filtros');
        const btnAtualizar = document.getElementById('btn-atualizar');

        if (btnFiltrar) {
            btnFiltrar.onclick = () => this.filtrarHistorico();
        }

        if (btnLimpar) {
            btnLimpar.onclick = () => this.limparFiltro();
        }

        if (btnAtualizar) {
            btnAtualizar.onclick = () => this.carregarHistorico();
        }
    }

    async carregarColaboradoresNoFiltro() {
        try {
            if (!this.api) return;

            const colaboradores = await this.api.getColaboradores();
            const selectFiltro = document.getElementById('filtroVendedor');
            
            if (!selectFiltro) {
                console.warn('Select filtroVendedor não encontrado');
                return;
            }
            
            // Limpar opções existentes (exceto a primeira)
            selectFiltro.innerHTML = '<option value="">Todos os vendedores</option>';
            
            // Adicionar colaboradores ao select
            colaboradores.forEach(colaborador => {
                const option = document.createElement('option');
                option.value = colaborador.id;
                option.textContent = `${colaborador.nome}`;
                selectFiltro.appendChild(option);
            });

            console.log(`✅ Carregados ${colaboradores.length} colaboradores no filtro`);
            
        } catch (error) {
            console.error('Erro ao carregar colaboradores no filtro:', error);
        }
    }

    async carregarHistorico(mes = null, resetPage = true) {
        try {
            const lista = document.getElementById('lista-comissoes');
            if (!lista) {
                console.error('Elemento lista-comissoes não encontrado');
                return;
            }

            lista.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"></div></div>';

            console.log('Chamando API para histórico com mês:', mes);
            const result = await this.api.getHistoricoComissoes(mes);
            console.log('Resultado da API:', result);

            if (result.success) {
                console.log('Dados recebidos:', result.data);
                
                // Armazenar todas as comissões
                this.allComissoes = result.data || [];
                this.totalItems = this.allComissoes.length;
                this.totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
                
                // Resetar para primeira página se necessário
                if (resetPage) {
                    this.currentPage = 1;
                }
                
                // Renderizar página atual
                this.renderizarPaginaAtual();
                this.renderizarControlesPaginacao();
                
            } else {
                console.log('Erro na API:', result.error);
                lista.innerHTML = `<div class="alert alert-danger">${result.error}</div>`;
            }

        } catch (error) {
            console.error('Erro ao carregar histórico:', error);
            const lista = document.getElementById('lista-comissoes');
            if (lista) {
                lista.innerHTML = '<div class="alert alert-danger">Erro ao carregar histórico</div>';
            }
        }
    }

    renderizarPaginaAtual() {
        const container = document.getElementById('lista-comissoes');
        if (!container) return;

        // Calcular índices da página atual
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const comissoesPagina = this.allComissoes.slice(startIndex, endIndex);

        console.log(`Renderizando página ${this.currentPage}: ${startIndex}-${endIndex}`, comissoesPagina);

        if (comissoesPagina.length === 0) {
            container.innerHTML = '<div class="alert alert-info">Nenhuma comissão encontrada</div>';
            return;
        }

        this.renderizarHistorico(comissoesPagina, container);
        this.renderizarControlesPaginacao();
    }

    renderizarHistorico(comissoes, container) {
        console.log('renderizarHistorico chamado com:', comissoes);
        console.log('Container:', container);
        
        if (!comissoes || comissoes.length === 0) {
            console.log('Nenhuma comissão encontrada');
            container.innerHTML = '<div class="alert alert-info">Nenhuma comissão encontrada</div>';
            return;
        }

        console.log(`Renderizando ${comissoes.length} comissões`);
        container.innerHTML = '';

        comissoes.forEach((comissao, index) => {
            console.log(`Renderizando comissão ${index}:`, comissao);
            const div = document.createElement('div');
            div.className = 'card mb-2';
            
            const dataRegistro = new Date(comissao.data_criacao);
            const agora = new Date();
            const diffHoras = (agora - dataRegistro) / (1000 * 60 * 60);
            const podeEditar = diffHoras <= 1; // 1 hora para edição

            div.innerHTML = `
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-8">
                            <h6 class="card-title">${comissao.colaborador_nome || 'Colaborador'}</h6>
                            <p class="card-text">
                                <strong>Mês:</strong> ${comissao.mes_referencia}<br>
                                <strong>PREMIO ISC:</strong> ${UIHelpers.formatCurrency(comissao.PREMIO_ISC)}<br>
                                <strong>PREMIAÇÃO META:</strong> ${UIHelpers.formatCurrency(comissao.PREMIACAO_META)}<br>
                                <strong>PREMIO:</strong> ${UIHelpers.formatCurrency(comissao.PREMIO)}<br>
                                <strong>GRATIFICAÇÃO:</strong> ${UIHelpers.formatCurrency(comissao.GRATIFICACAO)}
                            </p>
                            <small class="text-muted">Registrado em: ${UIHelpers.formatDate(comissao.data_criacao)}</small>
                        </div>
                        <div class="col-md-4">
                            ${podeEditar ? this.renderBotoesEdicao(comissao) : this.renderAvisoExpirado()}
                        </div>
                    </div>
                </div>
            `;

            container.appendChild(div);
        });
    }

    renderBotoesEdicao(comissao) {
        return `
            <button class="btn btn-sm btn-outline-primary" onclick="editarComissao(${comissao.id})">
                <i class="fas fa-edit"></i> Editar
            </button>
            <div id="edit-area-${comissao.id}" style="display:none;" class="mt-2">
                <div class="mb-2">
                    <label class="form-label fw-bold">PREMIO ISC:</label>
                    <input type="number" step="0.01" class="form-control form-control-sm" 
                           value="${comissao.PREMIO_ISC || ''}" id="edit-isc-${comissao.id}">
                </div>
                <div class="mb-2">
                    <label class="form-label fw-bold">PREMIAÇÃO META:</label>
                    <input type="number" step="0.01" class="form-control form-control-sm" 
                           value="${comissao.PREMIACAO_META || ''}" id="edit-meta-${comissao.id}">
                </div>
                <div class="mb-2">
                    <label class="form-label fw-bold">PREMIO:</label>
                    <input type="number" step="0.01" class="form-control form-control-sm" 
                           value="${comissao.PREMIO || ''}" id="edit-premio-${comissao.id}">
                </div>
                <div class="mb-2">
                    <label class="form-label fw-bold">GRATIFICAÇÃO:</label>
                    <input type="number" step="0.01" class="form-control form-control-sm" 
                           value="${comissao.GRATIFICACAO || ''}" id="edit-grat-${comissao.id}">
                </div>
                <button class="btn btn-sm btn-success" onclick="salvarEdicao(${comissao.id})">Salvar</button>
                <button class="btn btn-sm btn-secondary" onclick="cancelarEdicao(${comissao.id})">Cancelar</button>
                <div id="msg-edit-${comissao.id}" class="mt-2"></div>
            </div>
        `;
    }

    renderAvisoExpirado() {
        return '<div class="alert alert-warning alert-sm">Prazo de edição expirado (1h)</div>';
    }

    renderizarControlesPaginacao() {
        const container = document.getElementById('controles-paginacao');
        if (!container) return;

        if (this.totalPages <= 1) {
            container.innerHTML = '';
            return;
        }

        let html = '<nav aria-label="Navegação de páginas" class="mt-6">';
        html += '<div class="flex justify-center items-center space-x-1">';

        // Botão anterior
        html += `<button class="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-l-md hover:bg-gray-50 ${this.currentPage === 1 ? 'cursor-not-allowed opacity-50' : 'hover:text-gray-700'}" 
                        ${this.currentPage === 1 ? 'disabled' : ''} 
                        onclick="historico.irParaPagina(${this.currentPage - 1})">
                    <i class="fas fa-chevron-left"></i> Anterior
                 </button>`;

        // Números das páginas
        for (let i = 1; i <= this.totalPages; i++) {
            const isActive = i === this.currentPage;
            html += `<button class="px-3 py-2 text-sm font-medium ${isActive ? 'text-white bg-blue-600 border-blue-600' : 'text-gray-500 bg-white border-gray-300 hover:bg-gray-50 hover:text-gray-700'} border" 
                            onclick="historico.irParaPagina(${i})">
                        ${i}
                     </button>`;
        }

        // Botão próximo
        html += `<button class="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-r-md hover:bg-gray-50 ${this.currentPage === this.totalPages ? 'cursor-not-allowed opacity-50' : 'hover:text-gray-700'}" 
                        ${this.currentPage === this.totalPages ? 'disabled' : ''} 
                        onclick="historico.irParaPagina(${this.currentPage + 1})">
                    Próximo <i class="fas fa-chevron-right"></i>
                 </button>`;

        html += '</div>';
        html += `<div class="text-center text-gray-500 text-sm mt-3">
                    Página ${this.currentPage} de ${this.totalPages} 
                    (${this.totalItems} comissões no total)
                 </div>`;
        html += '</nav>';

        container.innerHTML = html;
    }

    irParaPagina(pagina) {
        if (pagina < 1 || pagina > this.totalPages) return;
        
        this.currentPage = pagina;
        this.renderizarPaginaAtual();
    }

    filtrarHistorico() {
        const mes = document.getElementById('filtroMes').value;
        const vendedor = document.getElementById('filtroVendedor').value;
        
        console.log('Filtrando por:', { mes, vendedor });
        
        // Por enquanto só implementamos filtro por mês
        // TODO: Implementar filtro por vendedor no backend
        this.carregarHistorico(mes, true); // Reset page quando filtrar
    }

    limparFiltro() {
        document.getElementById('filtroMes').value = '';
        document.getElementById('filtroVendedor').value = '';
        this.carregarHistorico(null, true); // Reset page quando limpar filtro
    }

    async salvarEdicao(comissaoId) {
        try {
            const dados = {
                PREMIO_ISC: this.parseFloat(document.getElementById(`edit-isc-${comissaoId}`).value),
                PREMIACAO_META: this.parseFloat(document.getElementById(`edit-meta-${comissaoId}`).value),
                PREMIO: this.parseFloat(document.getElementById(`edit-premio-${comissaoId}`).value),
                GRATIFICACAO: this.parseFloat(document.getElementById(`edit-grat-${comissaoId}`).value)
            };

            const result = await this.api.updateComissao(comissaoId, dados);

            if (result.success) {
                UIHelpers.showMessage(`msg-edit-${comissaoId}`, 'Comissão atualizada com sucesso!', 'success');
                setTimeout(() => this.carregarHistorico(), 1000);
            } else {
                UIHelpers.showMessage(`msg-edit-${comissaoId}`, result.error, 'error');
            }

        } catch (error) {
            console.error('Erro ao salvar edição:', error);
            UIHelpers.showMessage(`msg-edit-${comissaoId}`, 'Erro interno. Tente novamente.', 'error');
        }
    }

    parseFloat(value) {
        if (!value || value.trim() === '') return null;
        const num = parseFloat(value);
        return isNaN(num) ? null : num;
    }
}

// Funções globais para compatibilidade
window.editarComissao = function(id) {
    const editArea = document.getElementById(`edit-area-${id}`);
    editArea.style.display = editArea.style.display === 'none' ? 'block' : 'none';
};

window.cancelarEdicao = function(id) {
    document.getElementById(`edit-area-${id}`).style.display = 'none';
};

window.salvarEdicao = function(id) {
    window.historicoComissoes.salvarEdicao(id);
};

window.carregarHistoricoComissoes = function(mes = null) {
    window.historicoComissoes.carregarHistorico(mes);
};

// Inicializar quando DOM estiver pronto
document.addEventListener('DOMContentLoaded', function() {
    // Usar a mesma instância da API que é criada no HTML
    if (window.comissoesAPI) {
        window.registroComissao = new RegistroComissao(window.comissoesAPI);
        window.historicoComissoes = new HistoricoComissoes(window.comissoesAPI);
    } else {
        window.registroComissao = new RegistroComissao();
        window.historicoComissoes = new HistoricoComissoes();
    }
    
    // Carregar histórico inicial
    window.historicoComissoes.carregarHistorico();
});