/**
 * Classe para gerenciar relatórios de comissões
 */
class RelatoriosComissoes {
    constructor(api) {
        this.api = api;
        this.dadosOriginais = [];
        this.dadosFiltrados = [];
        this.paginaAtual = 1;
        this.itensPorPagina = 50;
        
        this.setupEventListeners();
    }

    setupEventListeners() {
        // Botões de ação
        document.getElementById('btn-filtrar-relatorio').addEventListener('click', () => {
            this.aplicarFiltros();
        });

        document.getElementById('btn-limpar-filtros-relatorio').addEventListener('click', () => {
            this.limparFiltros();
        });

        document.getElementById('btn-exportar-excel').addEventListener('click', () => {
            this.exportarParaExcel();
        });

        document.getElementById('btn-atualizar-relatorio').addEventListener('click', () => {
            this.carregarRelatorios();
        });

        // Carregar colaboradores no filtro
        this.carregarColaboradoresNoFiltro();
    }

    async carregarColaboradoresNoFiltro() {
        try {
            const colaboradores = await this.api.getColaboradores();
            const select = document.getElementById('filtroColaboradorRelatorio');
            
            // Limpar opções existentes (exceto a primeira)
            select.innerHTML = '<option value="">Todos os colaboradores</option>';
            
            // Adicionar colaboradores ao select
            colaboradores.forEach(colaborador => {
                const option = document.createElement('option');
                option.value = colaborador.id;
                option.textContent = `${colaborador.nome} - ${colaborador.departamento}`;
                select.appendChild(option);
            });

            console.log(`✅ Carregados ${colaboradores.length} colaboradores no filtro`);
            
        } catch (error) {
            console.error('Erro ao carregar colaboradores no filtro:', error);
        }
    }

    async carregarRelatorios() {
        try {
            this.mostrarLoading(true);
            
            console.log('📊 Carregando dados para relatório...');
            const result = await this.api.getHistoricoComissoes();
            
            if (result.success) {
                this.dadosOriginais = result.data || [];
                this.dadosFiltrados = [...this.dadosOriginais];
                console.log(`✅ Carregados ${this.dadosOriginais.length} registros`);
                
                this.atualizarEstatisticas();
                this.renderizarTabela();
                this.renderizarPaginacao();
            } else {
                console.error('❌ Erro ao carregar dados:', result.error);
                this.mostrarMensagem('Erro ao carregar dados dos relatórios');
            }
            
        } catch (error) {
            console.error('💥 Erro ao carregar relatórios:', error);
            this.mostrarMensagem('Erro interno ao carregar relatórios');
        } finally {
            this.mostrarLoading(false);
        }
    }

    aplicarFiltros() {
        const mes = document.getElementById('filtroMesRelatorio').value;
        const colaboradorId = document.getElementById('filtroColaboradorRelatorio').value;

        console.log('🔍 Aplicando filtros:', { mes, colaboradorId });

        this.dadosFiltrados = this.dadosOriginais.filter(item => {
            let passaFiltro = true;

            // Filtro por mês
            if (mes) {
                const dataItem = new Date(item.mes_referencia);
                const mesAno = `${dataItem.getFullYear()}-${String(dataItem.getMonth() + 1).padStart(2, '0')}`;
                passaFiltro = passaFiltro && (mesAno === mes);
            }

            // Filtro por colaborador
            if (colaboradorId) {
                passaFiltro = passaFiltro && (item.vendedor_id.toString() === colaboradorId);
            }

            return passaFiltro;
        });

        console.log(`✅ Filtros aplicados: ${this.dadosFiltrados.length} de ${this.dadosOriginais.length} registros`);

        this.paginaAtual = 1;
        this.atualizarEstatisticas();
        this.renderizarTabela();
        this.renderizarPaginacao();
    }

    limparFiltros() {
        document.getElementById('filtroMesRelatorio').value = '';
        document.getElementById('filtroColaboradorRelatorio').value = '';
        
        this.dadosFiltrados = [...this.dadosOriginais];
        this.paginaAtual = 1;
        
        this.atualizarEstatisticas();
        this.renderizarTabela();
        this.renderizarPaginacao();
        
        console.log('🧹 Filtros limpos');
    }

    atualizarEstatisticas() {
        const colaboradoresUnicos = new Set(this.dadosFiltrados.map(item => item.vendedor_id));
        const totalRegistros = this.dadosFiltrados.length;
        
        let totalGeral = 0;
        this.dadosFiltrados.forEach(item => {
            totalGeral += (item.PREMIO_ISC || 0) + (item.PREMIACAO_META || 0) + 
                         (item.PREMIO || 0) + (item.GRATIFICACAO || 0);
        });

        const mediaRegistro = totalRegistros > 0 ? totalGeral / totalRegistros : 0;

        // Atualizar elementos do DOM
        document.getElementById('total-colaboradores').textContent = colaboradoresUnicos.size;
        document.getElementById('total-registros').textContent = totalRegistros;
        document.getElementById('total-geral').textContent = this.formatarMoeda(totalGeral);
        document.getElementById('media-registro').textContent = this.formatarMoeda(mediaRegistro);
    }

    renderizarTabela() {
        const tbody = document.getElementById('tbody-relatorios');
        const inicio = (this.paginaAtual - 1) * this.itensPorPagina;
        const fim = inicio + this.itensPorPagina;
        const dadosPagina = this.dadosFiltrados.slice(inicio, fim);

        if (dadosPagina.length === 0) {
            this.mostrarMensagem('Nenhum registro encontrado para os filtros selecionados');
            tbody.innerHTML = '';
            return;
        }

        this.mostrarMensagem('', false);

        tbody.innerHTML = '';
        dadosPagina.forEach(item => {
            const row = this.criarLinhaTabela(item);
            tbody.appendChild(row);
        });
    }

    criarLinhaTabela(item) {
        const row = document.createElement('tr');
        row.className = 'hover:bg-gray-50';

        const premioIsc = item.PREMIO_ISC || 0;
        const premiacaoMeta = item.PREMIACAO_META || 0;
        const premio = item.PREMIO || 0;
        const gratificacao = item.GRATIFICACAO || 0;
        const total = premioIsc + premiacaoMeta + premio + gratificacao;

        row.innerHTML = `
            <td class="px-6 py-4 whitespace-nowrap">
                <div class="flex items-center">
                    <div class="flex-shrink-0 h-8 w-8">
                        <div class="h-8 w-8 rounded-full bg-primary text-white flex items-center justify-center text-sm font-medium">
                            ${item.colaborador_nome ? item.colaborador_nome.charAt(0).toUpperCase() : 'U'}
                        </div>
                    </div>
                    <div class="ml-3">
                        <div class="text-sm font-medium text-gray-900">
                            ${item.colaborador_nome || 'Nome não informado'}
                        </div>
                    </div>
                </div>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                ${this.formatarData(item.mes_referencia)}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                ${this.formatarMoeda(premiacaoMeta)}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                ${this.formatarMoeda(gratificacao)}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                ${this.formatarMoeda(premioIsc)}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                ${this.formatarMoeda(premio)}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                    ${this.formatarMoeda(total)}
                </span>
            </td>
        `;

        return row;
    }

    renderizarPaginacao() {
        const container = document.getElementById('paginacao-relatorios');
        const totalPaginas = Math.ceil(this.dadosFiltrados.length / this.itensPorPagina);

        if (totalPaginas <= 1) {
            container.innerHTML = '';
            return;
        }

        let html = '<nav class="flex items-center justify-between">';
        
        // Info da página
        html += `<div class="flex-1 flex justify-between sm:hidden">`;
        html += `<span class="text-sm text-gray-700">
                    Página ${this.paginaAtual} de ${totalPaginas}
                 </span>`;
        html += `</div>`;

        // Controles de navegação
        html += `<div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">`;
        html += `<div>
                    <p class="text-sm text-gray-700">
                        Mostrando <span class="font-medium">${(this.paginaAtual - 1) * this.itensPorPagina + 1}</span>
                        até <span class="font-medium">${Math.min(this.paginaAtual * this.itensPorPagina, this.dadosFiltrados.length)}</span>
                        de <span class="font-medium">${this.dadosFiltrados.length}</span> registros
                    </p>
                 </div>`;

        html += `<div class="flex space-x-2">`;
        
        // Botão anterior
        html += `<button onclick="window.relatorios.irParaPagina(${this.paginaAtual - 1})" 
                        ${this.paginaAtual === 1 ? 'disabled' : ''} 
                        class="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-md hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">
                    Anterior
                 </button>`;

        // Números das páginas (mostrar até 5 páginas)
        const inicio = Math.max(1, this.paginaAtual - 2);
        const fim = Math.min(totalPaginas, inicio + 4);

        for (let i = inicio; i <= fim; i++) {
            const isActive = i === this.paginaAtual;
            html += `<button onclick="window.relatorios.irParaPagina(${i})" 
                            class="px-3 py-2 text-sm font-medium ${isActive ? 'text-white bg-primary border-primary' : 'text-gray-500 bg-white border-gray-300 hover:bg-gray-50'} border rounded-md">
                        ${i}
                     </button>`;
        }

        // Botão próximo
        html += `<button onclick="window.relatorios.irParaPagina(${this.paginaAtual + 1})" 
                        ${this.paginaAtual === totalPaginas ? 'disabled' : ''} 
                        class="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-md hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">
                    Próximo
                 </button>`;

        html += `</div></div></nav>`;

        container.innerHTML = html;
    }

    irParaPagina(pagina) {
        const totalPaginas = Math.ceil(this.dadosFiltrados.length / this.itensPorPagina);
        
        if (pagina < 1 || pagina > totalPaginas) return;
        
        this.paginaAtual = pagina;
        this.renderizarTabela();
        this.renderizarPaginacao();
        
        // Scroll para o topo da tabela
        document.getElementById('tabela-relatorios').scrollIntoView({ behavior: 'smooth' });
    }

    exportarParaExcel() {
        try {
            console.log('📊 Iniciando exportação para Excel...');
            
            // Preparar dados para exportação
            const dadosExportacao = this.dadosFiltrados.map(item => {
                const premioIsc = item.PREMIO_ISC || 0;
                const premiacaoMeta = item.PREMIACAO_META || 0;
                const premio = item.PREMIO || 0;
                const gratificacao = item.GRATIFICACAO || 0;
                const total = premioIsc + premiacaoMeta + premio + gratificacao;

                return {
                    'Colaborador': item.colaborador_nome || 'Nome não informado',
                    'Data Referência': this.formatarDataExcel(item.mes_referencia),
                    'Premiação Meta': premiacaoMeta,
                    'Gratificação': gratificacao,
                    'Prêmio ISC': premioIsc,
                    'Prêmio': premio,
                    'Total': total
                };
            });

            // Criar workbook
            const wb = XLSX.utils.book_new();
            const ws = XLSX.utils.json_to_sheet(dadosExportacao);

            // Configurar largura das colunas
            const colWidths = [
                { wch: 25 }, // Colaborador
                { wch: 15 }, // Data
                { wch: 15 }, // Premiação Meta
                { wch: 15 }, // Gratificação
                { wch: 15 }, // Prêmio ISC
                { wch: 12 }, // Prêmio
                { wch: 15 }  // Total
            ];
            ws['!cols'] = colWidths;

            // Adicionar planilha ao workbook
            XLSX.utils.book_append_sheet(wb, ws, "Relatório de Comissões");

            // Gerar nome do arquivo
            const agora = new Date();
            const nomeArquivo = `Relatorio_Comissoes_${agora.getFullYear()}-${String(agora.getMonth() + 1).padStart(2, '0')}-${String(agora.getDate()).padStart(2, '0')}.xlsx`;

            // Salvar arquivo
            XLSX.writeFile(wb, nomeArquivo);
            
            console.log(`✅ Arquivo Excel exportado: ${nomeArquivo}`);
            
            // Mostrar mensagem de sucesso
            this.mostrarNotificacao('Relatório exportado com sucesso!', 'success');
            
        } catch (error) {
            console.error('💥 Erro ao exportar para Excel:', error);
            this.mostrarNotificacao('Erro ao exportar relatório. Tente novamente.', 'error');
        }
    }

    mostrarLoading(show) {
        const loading = document.getElementById('loading-relatorios');
        const tabela = document.getElementById('tabela-relatorios');
        
        if (show) {
            loading.classList.remove('hidden');
            tabela.style.opacity = '0.5';
        } else {
            loading.classList.add('hidden');
            tabela.style.opacity = '1';
        }
    }

    mostrarMensagem(mensagem, show = true) {
        const elemento = document.getElementById('mensagem-relatorios');
        
        if (show && mensagem) {
            elemento.innerHTML = `
                <i class="fas fa-info-circle text-4xl mb-2"></i>
                <p>${mensagem}</p>
            `;
            elemento.classList.remove('hidden');
        } else {
            elemento.classList.add('hidden');
        }
    }

    mostrarNotificacao(mensagem, tipo = 'info') {
        // Criar elemento de notificação
        const notificacao = document.createElement('div');
        notificacao.className = `fixed top-4 right-4 p-4 rounded-lg shadow-lg z-50 transition-all duration-300 transform translate-x-full`;
        
        if (tipo === 'success') {
            notificacao.className += ' bg-green-500 text-white';
        } else if (tipo === 'error') {
            notificacao.className += ' bg-red-500 text-white';
        } else {
            notificacao.className += ' bg-blue-500 text-white';
        }

        notificacao.innerHTML = `
            <div class="flex items-center">
                <i class="fas fa-${tipo === 'success' ? 'check' : tipo === 'error' ? 'exclamation' : 'info'}-circle mr-2"></i>
                <span>${mensagem}</span>
            </div>
        `;

        document.body.appendChild(notificacao);

        // Animar entrada
        setTimeout(() => {
            notificacao.classList.remove('translate-x-full');
        }, 100);

        // Remover após 3 segundos
        setTimeout(() => {
            notificacao.classList.add('translate-x-full');
            setTimeout(() => {
                if (notificacao.parentNode) {
                    notificacao.parentNode.removeChild(notificacao);
                }
            }, 300);
        }, 3000);
    }

    formatarMoeda(valor) {
        if (valor === null || valor === undefined || isNaN(valor)) {
            return 'R$ 0,00';
        }
        return new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(valor);
    }

    formatarData(dataString) {
        if (!dataString) return 'Data não informada';
        
        try {
            const data = new Date(dataString);
            return new Intl.DateTimeFormat('pt-BR', {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit'
            }).format(data);
        } catch {
            return 'Data inválida';
        }
    }

    formatarDataExcel(dataString) {
        if (!dataString) return 'Data não informada';
        
        try {
            const data = new Date(dataString);
            return `${String(data.getDate()).padStart(2, '0')}/${String(data.getMonth() + 1).padStart(2, '0')}/${data.getFullYear()}`;
        } catch {
            return 'Data inválida';
        }
    }
}